<?php

namespace App\Exports;


use App\Models\Pengeluaran;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithEvents;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Events\AfterSheet;

class PengeluaranExport implements FromCollection, WithHeadings, WithEvents, WithMapping
{
    protected $month;
    protected $years;
    protected $rowNumber = 0;
    protected $bulanIndonesia;
    public function __construct($month, $years, $bulanIndonesia)
    {
        $this->month = $month;
        $this->years = $years;
        $this->bulanIndonesia = $bulanIndonesia;
    }

    public function collection()
    {
        return Pengeluaran::select(
            'pengeluaran.*',
            'coa_debet.nama_akun as nama_akun_debet',
            'coa_kredit.nama_akun as nama_akun_kredit'
        )
            ->leftJoin('master_coa as coa_debet', 'coa_debet.no_akun', '=', 'pengeluaran.no_akun_debet')
            ->leftJoin('master_coa as coa_kredit', 'coa_kredit.no_akun', '=', 'pengeluaran.no_akun_kredit')
            ->whereYear('pengeluaran.tanggal', $this->years)
            ->whereMonth('pengeluaran.tanggal', $this->month)
            ->get();
    }

    public function map($row): array
    {
        $this->rowNumber++;

        return [
            $this->rowNumber,
            \Carbon\Carbon::parse($row->tanggal)->format('d/m/Y'),
            $row->nama_akun_kredit,
            $row->kredit,
            $row->nama_akun_debet,
            $row->debet,
            $row->keterangan
        ];
    }

    public function headings(): array
    {
        return [
            'No',
            'Tanggal',
            'Nama Akun',
            'Jumlah',
            'Nama Akun',
            'Jumlah',
            'Keterangan'
        ];
    }

    public function registerEvents(): array
    {
        return [
            AfterSheet::class => function (AfterSheet $event) {
                $sheet = $event->sheet->getDelegate();

                // === 1. Judul ===
                $sheet->insertNewRowBefore(1, 3);
                $sheet->setCellValue('A1', 'PT. Berkah Semangat Sentosa');
                $sheet->setCellValue('A2', 'Pengeluaran Bulan ' . $this->bulanIndonesia . ' Tahun ' . $this->years);
                $sheet->mergeCells('A1:G1');
                $sheet->mergeCells('A2:G2');
                $sheet->getStyle('A1:G2')->getAlignment()->setHorizontal('center');
                $sheet->getStyle('A1')->getFont()->setBold(true)->setSize(14);

                // === 2. Header Baris 3–4 ===
                $sheet->setCellValue('A3', 'No');
                $sheet->setCellValue('B3', 'Tanggal');
                $sheet->setCellValue('C3', 'Kredit');
                $sheet->setCellValue('E3', 'Debit');

                // Merge kolom Debit & Kredit
                $sheet->mergeCells('C3:D3'); // Kredit
                $sheet->mergeCells('E3:G3'); // Debit

                // Isi sub-kolom
                $sheet->setCellValue('C4', 'Nama Akun');
                $sheet->setCellValue('D4', 'Jumlah');
                $sheet->setCellValue('E4', 'Nama Akun');
                $sheet->setCellValue('F4', 'Jumlah');
                $sheet->setCellValue('G4', 'Keterangan');

                // Merge kolom No & Tanggal
                foreach (['A', 'B'] as $col) {
                    $sheet->mergeCells("{$col}3:{$col}4");
                }

                // === 3. Styling Header ===
                $sheet->getStyle('A3:G4')->getFont()->setBold(true);
                $sheet->getStyle('A3:G4')->getAlignment()->setHorizontal('center')->setVertical('center');
                $sheet->getStyle('A3:G4')->applyFromArray([
                    'borders' => [
                        'outline' => [
                            'borderStyle' => \PhpOffice\PhpSpreadsheet\Style\Border::BORDER_THIN,
                        ],
                        'inside' => [
                            'borderStyle' => \PhpOffice\PhpSpreadsheet\Style\Border::BORDER_THIN,
                        ],
                    ],
                ]);

                // === 4. Border Data ===
                $totalRows = 4 + $this->rowNumber;

                // Terapkan border ke seluruh data dari A5 sampai G{total}
                $sheet->getStyle("A5:G{$totalRows}")->applyFromArray([
                    'borders' => [
                        'outline' => [
                            'borderStyle' => \PhpOffice\PhpSpreadsheet\Style\Border::BORDER_THIN,
                        ],
                        'inside' => [
                            'borderStyle' => \PhpOffice\PhpSpreadsheet\Style\Border::BORDER_THIN,
                        ],
                    ],
                ]);
            }
        ];
    }
}
